package pro.spss.server.agent.utils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * 数据概要/字段类型快速生成工具。
 * 输入: 由 AgentFileReader 返回的原始样本内容，行之间用 /n 分隔，列之间用 /t 分隔。
 * 只解析前若干行样本生成列类型与示例，供大模型上下文嵌入，避免超长数据直接送入。
 */
public class DataSummaryUtil {

    private static final Pattern NUMERIC_PATTERN = Pattern.compile("^-?\\d+(\\.\\d+)?$");
    private static final int MAX_SAMPLE_VALUES_PER_COLUMN = 3;

    /**
     * 将 AgentFileReader 返回的样本原文转换为简明的数据概要文本。
     */
    public static String buildSummary(String rawSample) {
        if (rawSample == null || rawSample.isBlank()) {
            return "(无可用数据样本)";
        }
        // 分行（注意：AgentFileReader 使用的是字符串“/n”而非换行符）
        String[] lines = rawSample.split("/n");
        if (lines.length == 0) {
            return "(无数据行)";
        }
        // 表头（同理，使用字符串“/t”分隔）
        String[] headers = lines[0].split("/t");
        int colCount = headers.length;

        // 收集每列样本值 & 类型判断
        List<List<String>> columnValues = new ArrayList<>();
        for (int i = 0; i < colCount; i++) {
            columnValues.add(new ArrayList<>());
        }

        int dataRowCount = 0;
        for (int i = 1; i < lines.length; i++) {
            String[] cols = lines[i].split("/t", -1);
            for (int c = 0; c < colCount && c < cols.length; c++) {
                if (columnValues.get(c).size() < MAX_SAMPLE_VALUES_PER_COLUMN) {
                    String v = cols[c];
                    if (v != null && !v.isBlank()) {
                        columnValues.get(c).add(v);
                    }
                }
            }
            dataRowCount++;
        }

        // 生成概要文本
        StringBuilder sb = new StringBuilder();
        sb.append("【数据概要】\n");
        sb.append("列数:").append(colCount).append(" 样本行:").append(dataRowCount).append("\n");
        sb.append("列信息:\n");
        for (int i = 0; i < colCount; i++) {
            List<String> samples = columnValues.get(i);
            String type = inferType(samples);
            sb.append(i + 1).append(". ").append(headers[i]).append(" (类型:").append(type).append(") 示例:");
            if (samples.isEmpty()) {
                sb.append("无");
            } else {
                sb.append(String.join(", ", samples));
            }
            sb.append("\n");
        }

        // 粗略算法方向建议
        boolean hasNumeric = hasType(columnValues, "numeric");
        boolean hasCategorical = hasType(columnValues, "categorical");
        sb.append("可能适用的算法方向: ");
        if (hasNumeric && hasCategorical) {
            sb.append("分类(逻辑回归, 决策树, 随机森林), 回归(线性回归, 梯度提升), 特征工程(标准化, 缺失值处理)\n");
        } else if (hasNumeric) {
            sb.append("回归(最小二乘, 支持向量回归), 主成分/因子分析, 时间序列(AR/MA/ARMA)\n");
        } else if (hasCategorical) {
            sb.append("聚类(K均值/层次), 关联规则挖掘, 编码转换(哑编码, 二值化)\n");
        } else {
            sb.append("文本/灰色模型或需进一步的数据预处理\n");
        }
        return sb.toString();
    }

    private static String inferType(List<String> samples) {
        if (samples == null || samples.isEmpty()) {
            return "unknown";
        }
        boolean allNumeric = samples.stream().allMatch(v -> NUMERIC_PATTERN.matcher(v).matches());
        if (allNumeric) {
            return "numeric";
        }
        // 判断离散: 不重复值数量与样本数量比值
        Set<String> distinct = new HashSet<>(samples);
        if (distinct.size() <= 5 && distinct.size() <= samples.size()) {
            return "categorical";
        }
        return "text";
    }

    private static boolean hasType(List<List<String>> columnValues, String targetType) {
        for (List<String> samples : columnValues) {
            String type = inferType(samples);
            if (type.equals(targetType)) return true;
        }
        return false;
    }
}

