package pro.spss.server.agent.service.handler;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import pro.spss.server.agent.domain.constant.AlgoConfig;
import pro.spss.server.agent.domain.constant.ChatConstants;
import pro.spss.server.agent.domain.request.RequestParams;
import pro.spss.server.agent.domain.request.UserChatMessage;
import pro.spss.server.agent.service.chatCore.ChatHelper;
import pro.spss.server.agent.service.prompt.PromptBuilderService;
import pro.spss.server.agent.service.sessionService.ChatSessionManager;

/**
 * 意图识别
 */
@Service
@Slf4j
public class IntentTool {

    @Autowired
    private ChatHelper chatHelper;

    @Autowired
    private IntentHandlerRegistry intentHandlerRegistry;

    @Autowired
    private ChatSessionManager chatSessionManager;

    @Value("${agent.intent_history_limit:6}")
    private int intentHistoryLimit;

    @Value("${agent.handler_result}")
    private boolean handlerResult;

    @Autowired
    private PromptBuilderService promptBuilderService;

    /**
     * 意图识别：聚合系统提示词模板、动态工具清单、最近历史与用户本轮输入，
     * 调用大模型返回目标工具名称。默认回退为 "default"。
     * 模板中若包含 {{TOOLS}} 或 {{ALGO_LIST}}，将进行占位符替换。
     *
     * @param prompt          用户本轮原始提示
     * @param requestParams   会话上下文参数
     * @param userChatMessage 用户消息体
     * @return 工具名称（唯一 name）
     */
    public String getToolByIntent(String prompt, RequestParams requestParams, UserChatMessage userChatMessage) {
        // 读取系统提示词模板
        String intentPromptTemplate = promptBuilderService.build(AlgoConfig.INTENT_PROMPT);
        // 使用段落版本（当前生效的格式）；如需切换为 Markdown，只需改为 buildToolsMarkdownTable()
        String toolsText = buildToolsParagraph();
        // 将模板中的静态工具段替换为动态清单。模板中建议预留占位符 {{TOOLS}}
        String intentPrompt;
        if (intentPromptTemplate.contains("{{TOOLS}}")) {
            intentPrompt = intentPromptTemplate.replace("{{TOOLS}}", toolsText).replace("{{ALGO_LIST}}", AlgoConfig.ALGO_NAME_LIST);
        } else {
            // 向后兼容：如果模板没有占位符，则直接在末尾追加，避免破坏原文案
            intentPrompt = intentPromptTemplate + "\n\n" + toolsText;
        }

        String toolId = "default";
        String basePrompt = "是否确认执行：" + (requestParams.isConfirmed() ? "是" : "否") + "。\n";

        if (requestParams.getDataSummary() != null && !requestParams.getDataSummary().isBlank()) {
            prompt = ChatConstants.DATA_PROMPT + requestParams.getDataSummary() + ChatConstants.USER_PROMPT + prompt;
        }
        prompt = basePrompt + prompt;

        // 仅保留最近N条历史（用户/助手），不包含系统消息，N为配置 agent.intent_history_limit，默认5
        JSONArray fullHistory = chatSessionManager.getMessages(userChatMessage.getUserId());
        JSONArray history = new JSONArray();
        if (fullHistory != null && !fullHistory.isEmpty()) {
            int size = fullHistory.size();
            int limit = Math.max(0, intentHistoryLimit);
            int start = Math.max(0, size - limit);
            for (int i = start; i < size; i++) {
                history.add(fullHistory.get(i));
            }
        }
        log.info("历史指令条数：{}（已截断为最近{}条或更少）", history.size(), intentHistoryLimit);

        JSONArray messages = new JSONArray();
        messages.add(ChatConstants.createMessage(ChatConstants.ROLE_SYSTEM, intentPrompt));
        messages.addAll(history); // 添加最近历史消息（仅用户/助手）
        messages.add(ChatConstants.createMessage(ChatConstants.USER_ROLE, prompt));

        JSONObject requestBody = chatHelper.buildRequestBody(messages);
        String response = chatHelper.sendRequestSync(requestBody);
        JSONArray choices = chatHelper.parseResponse(response);

        if (choices != null && !choices.isEmpty()) {
            JSONObject assistantMessage = choices.getJSONObject(0).getJSONObject(ChatConstants.MESSAGE);
            String content = assistantMessage.getString(ChatConstants.CONTENT).replaceAll(ChatConstants.MARKDOWN, "");
            if (handlerResult) {
                content = content.substring(content.indexOf("</think>") + 10);
            }
            log.info("意图识别返回：{}", content);
            toolId = content;
        }
        return toolId;
    }

    /**
     * 构建工具清单（段落样式）：
     * 每个工具一段，包含 name/title/desc/condition（如有）/userExamplePrompt（如有），
     * 用于在系统提示词中让大模型感知当前已集成的工具。
     *
     * @return 段落样式的工具清单文本
     */
    private String buildToolsParagraph() {
        StringBuilder sb = new StringBuilder();
        intentHandlerRegistry.getAllToolInfos().forEach(m -> {
            String name = String.valueOf(m.getOrDefault("name", ""));
            String title = String.valueOf(m.getOrDefault("title", name));
            String desc = String.valueOf(m.getOrDefault("desc", ""));
            String example = String.valueOf(
                    m.getOrDefault("exampleUserPrompt",
                            m.getOrDefault("userExamplePrompt", ""))
            );
            String condition = String.valueOf(
                    m.getOrDefault("condition", "")
            );
            sb.append("name: ").append(name).append('\n')
                    .append("title: ").append(title).append('\n')
                    .append("desc: ").append(desc).append('\n')
                    .append("condition：").append(condition).append('\n');
            if (example != null && !example.isEmpty()) {
                sb.append("userExamplePrompt: ").append(example).append('\n');
            }
            sb.append('\n');
        });
        return sb.toString();
    }

    /**
     * 构建工具清单（Markdown 表格样式）：
     * 第一行表头，后续每行一个工具，包含 name/title/desc/userExamplePrompt。
     * 保留以便未来按配置切换展示样式。
     *
     * @return Markdown 表格样式的工具清单文本
     */
    private String buildToolsMarkdownTable() {
        StringBuilder sb = new StringBuilder();
        sb.append("name | title | desc | userExamplePrompt\n");
        sb.append("---|---|---|---\n");
        intentHandlerRegistry.getAllToolInfos().forEach(m -> {
            String name = String.valueOf(m.getOrDefault("name", ""));
            String title = String.valueOf(m.getOrDefault("title", name));
            String desc = String.valueOf(m.getOrDefault("desc", ""));
            String example = String.valueOf(
                    m.getOrDefault("exampleUserPrompt",
                            m.getOrDefault("userExamplePrompt", ""))
            );
            sb.append(name).append(" | ")
                    .append(title).append(" | ")
                    .append(desc).append(" | ")
                    .append(example)
                    .append('\n');
        });
        return sb.toString();
    }

}
