package pro.spss.server.agent.service.taskExecutor;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import pro.spss.server.agent.domain.DTO.TaskDTO;
import pro.spss.server.agent.domain.entity.ComputeResult;
import pro.spss.server.agent.domain.enums.ChatStatusEnum;
import pro.spss.server.agent.domain.enums.CreateWayEnum;
import pro.spss.server.agent.domain.enums.ResponseMessageType;
import pro.spss.server.agent.domain.enums.TaskStatusEnum;
import pro.spss.server.agent.domain.pojo.TaskMessage;
import pro.spss.server.agent.domain.request.RequestParams;
import pro.spss.server.agent.domain.response.ResponseMessage;
import pro.spss.server.agent.domain.response.Result;
import pro.spss.server.agent.mapper.ResultMapper;
import pro.spss.server.agent.service.feign.TaskServiceClient;
import pro.spss.server.agent.service.sseService.SseService;
import pro.spss.server.agent.utils.AlgoParamAutoGenerator;
import pro.spss.server.agent.utils.JsonUtil;
import pro.spss.server.agent.utils.TxtFileReader;

import java.io.File;
import java.util.List;

@Slf4j
@Service
public class TaskExecutorImpl implements TaskExecutor {

    @Autowired
    private SseService sseService;

    @Autowired
    private TaskServiceClient taskServiceClient;

    @Autowired
    private ResultMapper resultMapper;

    @Value("${path.analysis-result}")
    private String rootPath;


    @Override
    public ResponseMessage executeTask(RequestParams requestParams, String token) {
        // 若参数为空，尝试基于数据概要和算法ID自动生成一份默认参数
        if ((requestParams.getParams() == null || requestParams.getParams().isEmpty()) && requestParams.getAlgoId() != null) {
            JSONArray autoParams = AlgoParamAutoGenerator.generateDefaults(requestParams.getAlgoId(), requestParams.getDataSummary());
            if (!autoParams.isEmpty()) {
                requestParams.setParams(autoParams);
                log.info("Auto generated default params for algoId={} size={}", requestParams.getAlgoId(), autoParams.size());
            }
        }

        TaskDTO taskDTO = TaskDTO.builder()
                .algoId(requestParams.getAlgoId())
                .algoName(requestParams.getAlgoName())
                .dataId(requestParams.getDataId())
                .dataVersionId(requestParams.getDataVersionId())
                .createWay(CreateWayEnum.DATA_FIRST)
                .params(requestParams.getParams())
                .build();
        taskDTO.setTaskPriority(1);//TODO

        log.info(taskDTO.toString());
        Result taskResult = taskServiceClient.callExec(taskDTO, token);
        log.info("任务提交结果：{}", taskResult);
        if (taskResult == null || !taskResult.getSuccess()) {
//            sseService.sendMessage(Authenticator.currentUserId(), token, ChatStatusEnum.ALGORUNFAILED);
            ResponseMessage  responseMessage = new ResponseMessage();
            responseMessage.setCode(ChatStatusEnum.EXECUTOR_FAILED.getCode());
            responseMessage.setMessage(ChatStatusEnum.EXECUTOR_FAILED.getMessage() + (taskResult != null && taskResult.getMessage() != null ? taskResult.getMessage() : ""));
            responseMessage.setResponse(ChatStatusEnum.EXECUTOR_FAILED.getMessage());
            responseMessage.setType(ResponseMessageType.RESULT.getType());
//            SseServer.sendMessage(token, ChatStatusEnum.ALGORUNFAILED.toResult());
            return responseMessage;
        }

        return waitForTaskResult(taskResult, token);
    }

    private ResponseMessage waitForTaskResult(Result taskResult, String token) {
        int maxAttempts = 6;
        long interval = 5000;

        for (int i = 0; i < maxAttempts; i++) {
            try {
                TaskMessage taskMessage = resultMapper.getTaskMessage(taskResult.getData().toString());
                if (taskMessage == null) {
                    Thread.sleep(interval);
                    continue;
                }

                if (taskMessage.getTaskStatus() == TaskStatusEnum.VALIDATION_FAILED.getCode()
                        || taskMessage.getTaskStatus() == TaskStatusEnum.FAILED.getCode()) {
                    String resultPath = rootPath + taskMessage.getResultPath();
                    File resultFile = new File(resultPath);
                    if (resultFile.exists()) {
                        String errorFile = TxtFileReader.readSystemFile(resultPath);
                        JSONObject resultJson = new JSONObject();
                        resultJson.put("taskStatus", taskMessage.getTaskStatus());
                        resultJson.put("taskId", taskMessage.getTaskId());
                        resultJson.put("taskName", taskMessage.getTaskName());
                        resultJson.put("algoId", taskMessage.getAlgoId());
                        resultJson.put("resultJson", errorFile);
                        ResponseMessage  responseMessage = new ResponseMessage();
                        responseMessage.setCode(ChatStatusEnum.ALGORUNFAILED.getCode());
                        responseMessage.setType(ResponseMessageType.RESULT.getType());
                        responseMessage.setResult(resultJson);
                        responseMessage.setMessage("执行结果");
                        return responseMessage;
                    }
                }

                if (taskMessage.getTaskStatus() == TaskStatusEnum.SUCCESS.getCode()
                || taskMessage.getTaskStatus() == TaskStatusEnum.SUCCESS_WITH_WARNING.getCode()) {
                    String resultPath = rootPath + taskMessage.getResultPath();
                    File resultFile = new File(resultPath);

                    if (resultFile.exists()) {
                        List<ComputeResult> computeResults = JsonUtil.readJsonFromFile(resultPath);
                        log.debug(computeResults.toString());
                        JSONObject resultJson = new JSONObject();
                        resultJson.put("resultJson", computeResults);
                        resultJson.put("taskId", taskMessage.getTaskId());
                        resultJson.put("taskStatus", taskMessage.getTaskStatus());
                        resultJson.put("taskName", taskMessage.getTaskName());
                        resultJson.put("algoId", taskMessage.getAlgoId());
                        log.debug(resultJson.toString());
                        ResponseMessage  responseMessage = new ResponseMessage();
                        responseMessage.setCode(200);
                        responseMessage.setType(ResponseMessageType.RESULT.getType());
                        responseMessage.setResult(resultJson);
                        responseMessage.setMessage("执行结果");
//                        sseService.sendResult(Authenticator.currentUserId(), token, resultJson);
                        return responseMessage;
                    }
                }

                Thread.sleep(interval*i/2 + interval);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                ResponseMessage  responseMessage = new ResponseMessage();
                responseMessage.setCode(ChatStatusEnum.ALGORUNFAILED.getCode());
                responseMessage.setMessage(ChatStatusEnum.ALGORUNFAILED.getMessage());
                responseMessage.setResponse(ChatStatusEnum.ALGORUNFAILED.getMessage());
                responseMessage.setType(ResponseMessageType.RESULT.getType());
                e.printStackTrace();
                return responseMessage;
            } catch (Exception e) {
                ResponseMessage  responseMessage = new ResponseMessage();
                responseMessage.setCode(ChatStatusEnum.ALGORUNFAILED.getCode());
                responseMessage.setMessage(ChatStatusEnum.ALGORUNFAILED.getMessage());
                responseMessage.setResponse(ChatStatusEnum.ALGORUNFAILED.getMessage());
                responseMessage.setType(ResponseMessageType.RESULT.getType());
                e.printStackTrace();
                return responseMessage;
            }
        }
//        sseService.sendMessage(Authenticator.currentUserId(), token, ChatStatusEnum.NORESULT);
        ResponseMessage  responseMessage = new ResponseMessage();
        responseMessage.setCode(ChatStatusEnum.NORESULT.getCode());
        responseMessage.setMessage(ChatStatusEnum.NORESULT.getMessage());
        responseMessage.setResponse(ChatStatusEnum.NORESULT.getMessage());
        responseMessage.setType(ResponseMessageType.RESULT.getType());
        return responseMessage;
    }
}
