package pro.spss.server.agent.service.sseService;

import com.alibaba.fastjson2.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import pro.spss.server.agent.domain.enums.ChatStatusEnum;
import pro.spss.server.agent.domain.enums.ResponseMessageType;
import pro.spss.server.agent.domain.response.ResponseMessage;
import pro.spss.server.agent.service.sessionService.ChatSessionManager;
import pro.spss.server.common.interceptor.Authenticator;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.io.IOException;
import java.util.Map;
import java.util.concurrent.*;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Consumer;

@Slf4j
@Service
public class SseServiceImpl implements SseService {

    @Autowired
    private ChatSessionManager chatSessionManager;

    // 使用线程安全的Map存储用户连接
    private static final Map<String, SseEmitter> userEmitters = new ConcurrentHashMap<>();

    // 存储连接创建时间，用于监控和清理
    private static final Map<String, Long> connectionTimes = new ConcurrentHashMap<>();

    // 存储心跳任务
    private static final Map<String, ScheduledFuture<?>> heartbeatTasks = new ConcurrentHashMap<>();

    // 使用读写锁解决并发问题
    private static final ReadWriteLock lock = new ReentrantReadWriteLock();

    // 心跳线程池
    private static final ScheduledExecutorService heartbeatExecutor = Executors.newScheduledThreadPool(
            Runtime.getRuntime().availableProcessors() * 2,
            new ThreadFactory() {
                private final ThreadFactory defaultFactory = Executors.defaultThreadFactory();

                @Override
                public Thread newThread(Runnable r) {
                    Thread thread = defaultFactory.newThread(r);
                    thread.setName("sse-heartbeat-" + thread.getId());
                    thread.setDaemon(true);
                    return thread;
                }
            }
    );

    // 连接监控线程池
    private static final ScheduledExecutorService monitorExecutor = Executors.newSingleThreadScheduledExecutor(
            r -> {
                Thread t = new Thread(r, "sse-connection-monitor");
                t.setDaemon(true);
                return t;
            }
    );

    // 配置参数
    private static final long HEARTBEAT_INTERVAL = 30L; // 心跳间隔30秒
    private static final long CONNECTION_TIMEOUT = 30 * 60 * 1000L; // 连接超时30分钟
    private static final long MONITOR_INTERVAL = 5 * 60 * 1000L; // 监控间隔5分钟

    @PostConstruct
    public void init() {
        // 启动连接监控
        monitorExecutor.scheduleAtFixedRate(this::monitorConnections,
                MONITOR_INTERVAL, MONITOR_INTERVAL, TimeUnit.MILLISECONDS);
        log.info("SSE连接监控已启动，监控间隔: {}ms", MONITOR_INTERVAL);
    }

    @PreDestroy
    public void cleanup() {
        heartbeatExecutor.shutdown();
        monitorExecutor.shutdown();
        log.info("SSE服务已关闭");
    }

    /**
     * 监控连接状态，清理异常连接
     */
    private void monitorConnections() {
        try {
            long currentTime = System.currentTimeMillis();
            int expiredCount = 0;

            lock.writeLock().lock();
            try {
                for (Map.Entry<String, Long> entry : connectionTimes.entrySet()) {
                    String key = entry.getKey();
                    long createTime = entry.getValue();

                    // 检查连接是否超时
                    if (currentTime - createTime > CONNECTION_TIMEOUT) {
                        log.warn("连接超时，清理: {}", key);
                        cleanupConnection(key);
                        expiredCount++;
                    }
                }
            } finally {
                lock.writeLock().unlock();
            }

            if (expiredCount > 0) {
                log.info("连接监控清理了 {} 个超时连接", expiredCount);
            }

            // 记录当前连接数
            int activeConnections = userEmitters.size();
            if (activeConnections > 0) {
                log.info("当前活跃SSE连接数: {}", activeConnections);
            }

        } catch (Exception e) {
            log.error("连接监控异常: {}", e.getMessage(), e);
        }
    }

    @Override
    public SseEmitter connect() {
        String currentToken = String.valueOf(Authenticator.currentToken());
        String userId = Authenticator.currentUserId();
        String key = userId + "_" + currentToken;

        log.info("用户建立SSE连接: userId={}, key={}", userId, key);

        // 1. 清理已存在的连接
//        cleanupExistingConnection(key);

        // 2. 创建新的SSE连接
        SseEmitter emitter = new SseEmitter(CONNECTION_TIMEOUT);

        // 3. 保存连接信息
        lock.writeLock().lock();
        try {
            userEmitters.put(key, emitter);
            connectionTimes.put(key, System.currentTimeMillis());
        } finally {
            lock.writeLock().unlock();
        }

        // 4. 设置连接回调
        setupEmitterCallbacks(emitter, key, userId, currentToken);

        // 4.1 发送一次连接成功事件，便于前端确认建立并保持通道活动
        try {
            emitter.send(SseEmitter.event().name("connected").data("ok").id(String.valueOf(System.currentTimeMillis())));
        } catch (Exception e) {
            log.warn("初始connected事件发送失败: key={}, err={}", key, e.getMessage());
        }

        // 5. 启动心跳机制
        startHeartbeat(emitter, key);

        log.info("SSE连接建立成功: userId={}, 当前连接数={}", userId, userEmitters.size());
        return emitter;
    }

    /**
     * 清理已存在的连接
     */
    private void cleanupExistingConnection(String key) {
        lock.writeLock().lock();
        try {
            SseEmitter oldEmitter = userEmitters.get(key);
            if (oldEmitter != null) {
                log.info("清理已存在的连接: {}", key);
                try {
                    // 先发送关闭通知
                    oldEmitter.send(SseEmitter.event()
                            .name("close")
                            .data("新连接建立，旧连接关闭")
                            .reconnectTime(0));
                } catch (Exception e) {
                    // 忽略发送异常
                }

                try {
                    oldEmitter.complete();
                } catch (Exception e) {
                    log.debug("完成旧连接异常: {}", e.getMessage());
                }

                // 从集合中移除
                userEmitters.remove(key);
                connectionTimes.remove(key);
            }

            // 清理心跳任务
            ScheduledFuture<?> oldTask = heartbeatTasks.remove(key);
            if (oldTask != null) {
                oldTask.cancel(true);
                log.debug("取消旧连接的心跳任务: {}", key);
            }
        } finally {
            lock.writeLock().unlock();
        }
    }

    private void trySendErrorEvent(SseEmitter emitter, String message) {
        try {
            if (emitter != null) {
                emitter.send(SseEmitter.event().name("error").data(message).id(String.valueOf(System.currentTimeMillis())));
            }
        } catch (Exception ignore) {
            // 忽略发送错误事件的异常，继续清理
        }
    }

    /**
     * 设置连接回调
     */
    private void setupEmitterCallbacks(SseEmitter emitter, String key, String userId, String token) {
        emitter.onCompletion(() -> {
            log.info("SSE连接完成: userId={}, key={}", userId, key);
            cleanupConnection(key);
        });

        emitter.onTimeout(() -> {
            log.info("SSE连接超时: userId={}, key={}", userId, key);
            trySendErrorEvent(emitter, "timeout");
            cleanupConnection(key);
        });

        emitter.onError(throwable -> {
            log.error("SSE连接错误: userId={}, key={}, error={}",
                    userId, key, throwable.getMessage(), throwable);
            trySendErrorEvent(emitter, "error: " + (throwable == null ? "unknown" : throwable.getMessage()));
            cleanupConnection(key);
        });
    }

    /**
     * 启动心跳机制
     */
    private void startHeartbeat(SseEmitter emitter, String key) {
        ScheduledFuture<?> heartbeatFuture = heartbeatExecutor.scheduleAtFixedRate(() -> {
            try {
                lock.readLock().lock();
                try {
                    SseEmitter currentEmitter = userEmitters.get(key);
                    if (currentEmitter != null && currentEmitter.equals(emitter)) {
                        // 发送心跳消息
                        currentEmitter.send(SseEmitter.event()
                                .name("heartbeat")
                                .data("ping")
                                .id(String.valueOf(System.currentTimeMillis())));
                        log.debug("发送心跳成功: key={}", key);
                    } else {
                        // 连接已被清理
                        ScheduledFuture<?> task = heartbeatTasks.remove(key);
                        if (task != null) {
                            task.cancel(false);
                        }
                    }
                } finally {
                    lock.readLock().unlock();
                }
            } catch (IOException e) {
                log.warn("心跳发送失败，连接可能已断开: key={}, error={}", key, e.getMessage());
                cleanupConnection(key);
            } catch (Exception e) {
                log.error("心跳任务异常: key={}, error={}", key, e.getMessage(), e);
            }
        }, 5, HEARTBEAT_INTERVAL, TimeUnit.SECONDS);

        heartbeatTasks.put(key, heartbeatFuture);
    }

    /**
     * 清理连接资源
     */
    private void cleanupConnection(String key) {
        lock.writeLock().lock();
        try {
            SseEmitter emitter = userEmitters.remove(key);
            if (emitter != null) {
                try {
                    emitter.complete();
                } catch (Exception e) {
                    log.debug("完成emitter异常: {}", e.getMessage());
                }
            }

            connectionTimes.remove(key);

            ScheduledFuture<?> task = heartbeatTasks.remove(key);
            if (task != null && !task.isCancelled()) {
                task.cancel(true);
            }

            log.info("清理SSE连接完成: key={}, 剩余连接数={}", key, userEmitters.size());
        } finally {
            lock.writeLock().unlock();
        }
    }

    @Override
    public String sendResult(String userId, String userToken, JSONObject jsonObject) {
        String type = "agent_message";
        return sendToUser(type, userId, userToken, messageBody -> {
            messageBody.setType(ResponseMessageType.RESULT.getType());
            messageBody.setResult(jsonObject);
            messageBody.setMessage("执行结果");
        });
    }


    @Override
    public String sendMessage(String type, String userId, String userToken, String message) {
        return sendToUser("agent_message", userId, userToken, messageBody -> {
            messageBody.setType(type);
            messageBody.setMessage(message);
        });
    }

    @Override
    public String sendMessage(String userId, String userToken, ResponseMessage message) {
        return sendToUser("agent_message", userId, userToken, messageBody -> {
            messageBody.setType(message.getType().isBlank() ?
                    ResponseMessageType.MESSAGE.getType() : message.getType());
            messageBody.setResponse(message.getResponse() == null ? "" : message.getResponse());
            messageBody.setOptions(message.getOptions());
            messageBody.setCode(message.getCode());
            messageBody.setStartTimestamp(message.getStartTimestamp());
            messageBody.setEndTimestamp(message.getEndTimestamp());
            messageBody.setMessage(message.getMessage() == null ? "" : message.getMessage());
            messageBody.setMessageId(message.getMessageId());
            messageBody.setSessionId(message.getSessionId());
            if (message.getResult() != null) {
                messageBody.setResult(message.getResult());
            }
        });
    }

    @Override
    public String sendMessage(String userId, String userToken, ChatStatusEnum message) {
        return sendToUser("agent_message", userId, userToken, messageBody -> {
            messageBody.setType(ResponseMessageType.MESSAGE.getType());
            messageBody.setCode(message.getCode());
            messageBody.setResponse(message.getMessage());
        });
    }

    @Override
    public String sendMessage(String userId, String userToken, String message) {
        return sendMessage(ResponseMessageType.MESSAGE.getType(), userId, userToken, message);
    }

    /**
     * 核心发送方法 - 带有重试机制的增强版
     */
    private String sendToUser(String eventName, String userId, String userToken,
                              Consumer<ResponseMessage> messageCustomizer) {
        String key = userId + "_" + userToken;
        log.debug("尝试发送消息: userId={}, event={}, key={}", userId, eventName, key);

        // 1. 检查连接是否存在
        SseEmitter emitter = null;
        lock.readLock().lock();
        try {
            emitter = userEmitters.get(key);
        } finally {
            lock.readLock().unlock();
        }

        if (emitter == null) {
            log.warn("发送失败: 用户连接不存在, userId={}, key={}", userId, key);
            return "目标用户未建立SSE连接或已离线";
        }

        // 2. 构建消息
        ResponseMessage msg = new ResponseMessage();
        msg.setUserId(userId);
        messageCustomizer.accept(msg);

        // 3. 发送消息（带重试机制）
        int maxRetries = 3;
        for (int retry = 0; retry < maxRetries; retry++) {
            try {
                emitter.send(SseEmitter.event()
                        .name(eventName)
                        .data(msg)
                        .id(String.valueOf(System.currentTimeMillis())));

                log.debug("消息发送成功: userId={}, event={}, retry={}", userId, eventName, retry);
                return "消息发送成功";

            } catch (IOException e) {
                log.warn("发送失败(第{}次重试): userId={}, error={}",
                        retry + 1, userId, e.getMessage());

                if (retry == maxRetries - 1) {
                    // 最后一次重试失败，清理连接
                    log.error("消息发送失败，清理连接: userId={}, key={}", userId, key);
                    cleanupConnection(key);
                    return "发送失败，连接已断开";
                }

                // 等待后重试
                try {
                    Thread.sleep(100 * (retry + 1)); // 递增等待时间
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    return "发送中断";
                }

            } catch (IllegalStateException e) {
                // 连接状态异常
                log.error("连接状态异常: userId={}, error={}", userId, e.getMessage());
                cleanupConnection(key);
                return "连接状态异常";
            } catch (Exception e) {
                log.error("发送消息异常: userId={}, error={}", userId, e.getMessage(), e);
                return "发送异常";
            }
        }

        return "发送失败";
    }

    /**
     * 获取当前活跃连接数
     */
    public int getActiveConnectionCount() {
        lock.readLock().lock();
        try {
            return userEmitters.size();
        } finally {
            lock.readLock().unlock();
        }
    }

    /**
     * 强制关闭指定用户的连接
     */
    public void forceCloseConnection(String userId, String userToken) {
        String key = userId + "_" + userToken;
        log.info("强制关闭连接: key={}", key);
        cleanupConnection(key);
    }
}

