package pro.spss.server.agent.service.handler.tools;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import pro.spss.server.agent.domain.constant.AlgoConfig;
import pro.spss.server.agent.domain.constant.ChatConstants;
import pro.spss.server.agent.domain.entity.AiChatResponse;
import pro.spss.server.agent.domain.request.RequestParams;
import pro.spss.server.agent.domain.response.ResponseMessage;
import pro.spss.server.agent.service.handler.ConversationSupport;
import pro.spss.server.agent.service.prompt.PromptBuilderService;

/**
 * 系统推荐算法处理器
 */
@Slf4j
@Component
public class SuggestIntentHandler implements IntentStateHandler {

    private static final Logger logger = LoggerFactory.getLogger(SuggestIntentHandler.class);

    @Autowired
    private PromptBuilderService promptBuilderService;

    @Autowired
    private ConversationSupport support;

    @Override
    public String getName() { return "suggest_algorithm"; }

    @Override
    public String getTitle() { return "算法推荐工具"; }

    @Override
    public String getDesc() { return "能够根据用户询问的问题以及携带的数据概览进行算法推荐功能。"; }

    @Override
    public String exampleUserPrompt() {
        return "对31个省市的创新效率进行评价";
    }

    @Override
    public String getCondition() {
        return "是否上传数据为是";
    }

    @Override
    public ResponseMessage handle(AiChatResponse intentResult, RequestParams requestParams, JSONArray historyMessages, String prompt, String token) {
        logger.info("系统推荐算法触发");
        JSONArray algoOptions = new JSONArray();
        if (intentResult.getRecommendedAlgorithms() != null && !intentResult.getRecommendedAlgorithms().isEmpty()) {
            algoOptions.addAll(intentResult.getRecommendedAlgorithms());
        }
        if (intentResult.getAlgoName() != null && !intentResult.getAlgoName().isEmpty()) {
            algoOptions.add(intentResult.getAlgoName());
        }

        if (algoOptions.isEmpty()) {//上一步没有推荐出来算法
            String intentPrompt = promptBuilderService.build(AlgoConfig.RECOGNIZE_ALGO_NAME);
            if(historyMessages.isEmpty()) {
                JSONArray messages = new JSONArray();
                messages.add(ChatConstants.createMessage(ChatConstants.ROLE_SYSTEM, intentPrompt));
                messages.add(ChatConstants.createMessage(ChatConstants.USER_ROLE, prompt));
                historyMessages.addAll(messages);
            } else {
                historyMessages.add(0, ChatConstants.createMessage(ChatConstants.ROLE_SYSTEM, intentPrompt));
                historyMessages.add(ChatConstants.createMessage(ChatConstants.USER_ROLE, prompt));
            }
            //构建请求大模型参数，并请求
            ResponseMessage responseMessage = support.sendRequestWithMessages(historyMessages, requestParams);
            log.info("推荐算法结果：" + JSONObject.toJSONString(responseMessage));
            return responseMessage;
        }
        ResponseMessage msg = new ResponseMessage();
        msg.setResponse(intentResult.getResponse() != null ? intentResult.getResponse() : ChatConstants.WELCOME_MESSAGE);
        msg.setOptions(algoOptions);
        return msg;
    }
}
