package pro.spss.server.agent.service.handler;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import pro.spss.server.agent.domain.constant.ChatConstants;
import pro.spss.server.agent.domain.entity.AIResponse;
import pro.spss.server.agent.domain.enums.ChatStatusEnum;
import pro.spss.server.agent.domain.enums.ConversationStateEnum;
import pro.spss.server.agent.domain.request.RequestParams;
import pro.spss.server.agent.domain.response.ResponseMessage;
import pro.spss.server.agent.service.chatCore.ChatHelper;

/**
 * 提供 ConversationHandler 可复用的支撑方法，便于被状态处理器调用。
 */
@Slf4j
@Component
public class ConversationSupport {

    private final ChatHelper chatHelper;

    @Value("${agent.chat.retry.maxAttempts:2}")
    private int maxAttempts;
    @Value("${agent.chat.retry.backoffMs:500}")
    private long backoffMs;
    @Value("${agent.chat.timeout.totalMs:10000}")
    private long totalTimeoutMs;

    public ConversationSupport(ChatHelper chatHelper) {
        this.chatHelper = chatHelper;
    }

    /**
     * 将系统提示与用户输入注入历史，构造请求，统一重试与错误处理，返回标准响应。
     */
    public ResponseMessage sendRequestWithHistoryMessages(JSONArray historyMessages, String intentPrompt, String prompt) {
        // 注入系统与用户消息
        if (historyMessages == null) {
            historyMessages = new JSONArray();
        }
        if (historyMessages.isEmpty()) {
            JSONArray messages = new JSONArray();
            messages.add(ChatConstants.createMessage(ChatConstants.ROLE_SYSTEM, intentPrompt));
            messages.add(ChatConstants.createMessage(ChatConstants.USER_ROLE, prompt));
            historyMessages.addAll(messages);
        } else {
            historyMessages.add(0, ChatConstants.createMessage(ChatConstants.ROLE_SYSTEM, intentPrompt));
            historyMessages.add(ChatConstants.createMessage(ChatConstants.USER_ROLE, prompt));
        }

        JSONObject requestBody = chatHelper.buildRequestBody(historyMessages);
        String response = sendWithRetry(requestBody);
        if (response == null) {
            return buildFailed(ChatStatusEnum.REQUESTFAILED);
        }
        AIResponse ai = safeParse(response);
        if (ai == null) {
            return buildFailed(ChatStatusEnum.RESPONSEFAILED);
        }
        ResponseMessage msg = new ResponseMessage();
        msg.setResponse(ai.getResponseText());
        msg.setThink(ai.getThink());
        msg.setTotalToken(ai.getTotalToken());
        return msg;
    }

    /**
     * 直接使用给定消息数组发起请求，统一重试、解析与响应映射，同时更新必要的 RequestParams 状态。
     */
    public ResponseMessage sendRequestWithMessages(JSONArray messages, RequestParams requestParams) {
        if (messages == null) messages = new JSONArray();
        JSONObject requestBody = chatHelper.buildRequestBody(messages);
        String responseBody = sendWithRetry(requestBody);
        if (responseBody == null) {
            return buildFailed(ChatStatusEnum.REQUESTFAILED);
        }
        // 使用 ChatHelper 的纯解析，提取 response/params/options
        JSONObject contentJson = chatHelper.parseContentJson(responseBody);
        if (contentJson == null) {
            return buildFailed(ChatStatusEnum.RESPONSEFAILED);
        }
        String response = contentJson.getString(ChatConstants.RESPONSE);
        JSONArray params = contentJson.getJSONArray(ChatConstants.PARAMS);
        JSONArray options = contentJson.getJSONArray(ChatConstants.OPTIONS);
        if (requestParams != null && params != null && !params.isEmpty()) {
            requestParams.updateParams(params);
            requestParams.setState(ConversationStateEnum.PARAM_CONFIRMATION);
        }
        ResponseMessage msg = new ResponseMessage();
        msg.setResponse(response);
        msg.setOptions(options != null ? options : new JSONArray());
        return msg;
    }

    /**
     * 安全解析为 AIResponse；失败时返回 null。
     */
    private AIResponse safeParse(String response) {
        try {
            return chatHelper.parseResponseText(response);
        } catch (Exception e) {
            log.warn("解析响应到 AIResponse 失败：{}", e.getMessage());
            return null;
        }
    }

    /**
     * 统一构造失败响应。
     */
    private ResponseMessage buildFailed(ChatStatusEnum status) {
        ResponseMessage msg = new ResponseMessage();
        msg.setCode(status.getCode());
        msg.setMessage(status.getMessage());
        msg.setResponse(status.getMessage());
        msg.setOptions(new JSONArray());
        return msg;
    }

    private String sendWithRetry(JSONObject requestBody) {
        return chatHelper.sendWithRetry(requestBody, maxAttempts, backoffMs, totalTimeoutMs);
    }
}
