package pro.spss.server.agent.service.chatService;

import com.alibaba.fastjson2.JSONArray;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import pro.spss.server.agent.domain.enums.ChatStatusEnum;
import pro.spss.server.agent.domain.enums.ConversationStateEnum;
import pro.spss.server.agent.domain.enums.CreateWayEnum;
import pro.spss.server.agent.domain.enums.ResponseMessageType;
import pro.spss.server.agent.domain.request.RequestParams;
import pro.spss.server.agent.domain.request.UserChatMessage;
import pro.spss.server.agent.domain.response.ResponseMessage;
import pro.spss.server.agent.domain.response.Result;
import pro.spss.server.agent.service.handler.ConversationHandler;
import pro.spss.server.agent.service.handler.IntentTool;
import pro.spss.server.agent.service.sessionService.ChatSessionManager;
import pro.spss.server.agent.service.sseService.SseService;
import pro.spss.server.agent.utils.AgentFileReader;
import pro.spss.server.agent.utils.DataSummaryUtil;

@Slf4j
@Service
public class ChatServiceImpl implements BaseChatService {

    @Autowired
    private ChatSessionManager chatSessionManager;

    @Value("${agent.init_prompt}")
    private String initPrompt;

    @Autowired
    private AgentFileReader agentFileReader;

    @Autowired
    private ConversationHandler conversationHandler;

    @Autowired
    private SseService sseService;

    @Autowired
    private IntentTool intentTool;

    @Async
    @Override
    public void baseChat(UserChatMessage userChatMessage, boolean retry) {
        if (handleFirstRequestIfNeeded(userChatMessage)) {
            return;
        }

        long startTimestamp = System.currentTimeMillis();
        ResponseMessage responseMessage = new ResponseMessage();
        try {
            RequestParams requestParams = prepareContext(userChatMessage);
            String toolName = detectIntent(userChatMessage, requestParams); //调用意图识别工具去识别用户意图，以及返回应该调用的工具名称
            responseMessage = executeTool(userChatMessage, requestParams, toolName, startTimestamp); //根据识别到的工具名称去调用对应的工具执行
            // 推送SSE
            sendSse(userChatMessage, responseMessage);

        } catch (Exception e) {
            e.printStackTrace();
            handleFailure(e, userChatMessage, responseMessage);
        }
    }

    /**
     * 是否为首次请求：
     * - 若用户对应 session 不存在，则初始化会话并通过 SSE 推送 initPrompt
     * - 返回 true 表示已处理（baseChat 需要直接 return）
     */
    private boolean handleFirstRequestIfNeeded(UserChatMessage userChatMessage) {
        String sessionId = userChatMessage.getUserId();
        if (sessionId == null || sessionId.isBlank()) {
            return false;
        }

        if (chatSessionManager.getMessages(sessionId) != null) {
            return false;
        }

        chatSessionManager.initSession(sessionId, CreateWayEnum.ALGO_FIRST);

        sseService.sendMessage(ResponseMessageType.INIT.getType(), sessionId, userChatMessage.getToken(), initPrompt);
        return true;
    }

    /**
     * 准备上下文：从 ChatSessionManager 取出/创建 RequestParams，
     * 写入用户确认状态和数据概要（若用户上传了数据则读取样本并生成摘要），用于后续意图识别与工具执行。
     *
     * @param userChatMessage 用户聊天请求（含 userId、prompt、confirm、dataId 等）
     * @return 本次会话的上下文参数（包含确认标记、数据摘要、数据版本等）
     */
    private RequestParams prepareContext(UserChatMessage userChatMessage) {
        String userId = userChatMessage.getUserId();
        RequestParams requestParams = chatSessionManager.getRequestParams(userId);
        requestParams.setConfirmed(userChatMessage.getConfirm());
        log.debug("==================userChatMessage=================");
        log.debug(userChatMessage.toString());
        handlerData(requestParams, userChatMessage);
        return requestParams;
    }

    /**
     * 意图识别：构造含工具清单与必要上下文的提示词，调用大模型进行意图判定，返回工具的唯一 name。
     *
     * @param userChatMessage 用户消息体（用于拼接提示词与历史）
     * @param requestParams   会话上下文参数（确认标记、数据摘要等）
     * @return 识别得到的工具名称（IntentStateHandler.name），默认返回 "default"
     */
    private String detectIntent(UserChatMessage userChatMessage, RequestParams requestParams) {
        String prompt = userChatMessage.getPrompt();
        return intentTool.getToolByIntent(prompt, requestParams, userChatMessage);
    }

    /**
     * 执行工具：以当前会话的最近历史、用户输入、意图识别的工具名为参数，
     * 调用对应的工具处理器，返回构造完毕的 ResponseMessage（含 userId、sessionId、起止时间戳）。
     *
     * @param userChatMessage 用户请求入参
     * @param requestParams   会话上下文参数
     * @param toolName        意图识别得到的工具名称
     * @param startTimestamp  工具执行开始毫秒时间戳
     * @return 工具执行的响应消息
     */
    private ResponseMessage executeTool(UserChatMessage userChatMessage, RequestParams requestParams, String toolName, long startTimestamp) {
        String sessionId = userChatMessage.getUserId();
        JSONArray historyCopy = new JSONArray(chatSessionManager.getMessages(sessionId));
        String prompt = userChatMessage.getPrompt();
        ResponseMessage responseMessage = conversationHandler.toolExecutor(requestParams, userChatMessage, toolName, historyCopy, prompt);
        responseMessage.setStartTimestamp(startTimestamp);
        responseMessage.setEndTimestamp(System.currentTimeMillis());
        responseMessage.setUserId(userChatMessage.getUserId());
        responseMessage.setSessionId(userChatMessage.getUserId());
        return responseMessage;
    }

    /**
     * 推送 SSE 消息：沿用以 userId 为通道的推送方式，响应体中包含 sessionId，前端可按会话路由展示。
     *
     * @param userChatMessage 用户消息（包含 userId 与推送 token）
     * @param responseMessage 响应消息
     */
    private void sendSse(UserChatMessage userChatMessage, ResponseMessage responseMessage) {
        sseService.sendMessage(userChatMessage.getUserId(), userChatMessage.getToken(), responseMessage);
    }

    /**
     * 异常兜底：当任一步骤抛出异常时，返回统一的失败响应并进行 SSE 推送，同时写入错误日志。
     *
     * @param e               异常
     * @param userChatMessage 用户消息
     * @param responseMessage 已构造的响应消息（补充失败信息）
     */
    private void handleFailure(Exception e, UserChatMessage userChatMessage, ResponseMessage responseMessage) {
        responseMessage.setResponse("似乎出现了点问题，我的大脑有点出小差了，请稍后再一起探讨吧。");
        responseMessage.setCode(ChatStatusEnum.REQUESTFAILED.getCode());
        log.error("BaseChatImpl.baseChat 处理异常：", e);
        sseService.sendMessage(userChatMessage.getUserId(), userChatMessage.getToken(), responseMessage);
    }

    @Override
    public Result clearHistory(String sessionId) {
        chatSessionManager.clear(sessionId);
        return Result.success();
    }

    /**
     * 会话上下文数据处理：
     * - 若存在 dataId：切换会话状态为意图识别，读取数据样本并生成摘要，保存到 RequestParams
     * - 若存在 dataVersionId：写入数据版本信息
     *
     * @param requestParams   会话上下文参数
     * @param userChatMessage 用户消息体
     * @return 处理后的上下文参数
     */
    private RequestParams handlerData(RequestParams requestParams, UserChatMessage userChatMessage) {
        String dataId = userChatMessage.getDataId();
        String dataVersionId = userChatMessage.getDataVersionId();
        log.debug("==================handlerData=================");
        log.debug("dataId: " + dataId);
        log.debug("dataVersionId: " + dataVersionId);
        if (dataId != null && !dataId.isBlank()) {
            try {
                requestParams.setState(ConversationStateEnum.INTENT_RECOGNITION);
                requestParams.updateDataId(dataId);

                String rawSample = agentFileReader.readCsvFileByDataId(dataId, userChatMessage.getUserId());
                String data = DataSummaryUtil.buildSummary(rawSample);
                requestParams.setDataSummary(data); // 保存数据概要供后续自动参数生成使用
                log.debug("Loaded data sample for dataId={} summary length={}", dataId, data.length());
            } catch (Exception e) {
                log.warn("Failed reading data for dataId {}: {}", dataId, e.getMessage());
            }
        }

        if (dataVersionId != null && !dataVersionId.isBlank()) {
            requestParams.updateDataVersionId(dataVersionId);
        }
        log.debug(requestParams.toString());
        return requestParams;
    }

    /**
     * 根据用户提问生成简短会话标题：去除换行、压缩空白、长度截断（默认30），
     * 返回空字符串时用 null 表示不更新。
     *
     * @param prompt 用户本轮提问文本
     * @return 生成的简短标题或 null
     */
    private String generateTitleFromPrompt(String prompt) {
        if (prompt == null) return null;
        String p = prompt.trim();
        // 去换行、压缩空白
        p = p.replaceAll("\r?\n+", " ").replaceAll("\\s+", " ");
        int maxLen = 30; // 截断长度，可按需改成配置
        if (p.length() > maxLen) {
            p = p.substring(0, maxLen) + "...";
        }
        return p.isEmpty() ? null : p;
    }

    @Override
    public String testIntent(UserChatMessage userChatMessage) {
        String userMessage = userChatMessage.getPrompt();
        log.info("用户闲聊：" + userMessage);

        // 使用用户ID作为SSE连接/缓存的键，但严格按会话ID进行持久化与缓存判断
        String userId = userChatMessage.getUserId();
        String requestSessionIdStr = userChatMessage.getUserId();
        if (requestSessionIdStr == null || requestSessionIdStr.isBlank()) {
            log.warn("UserChatMessage.sessionId 为空，无法准确路由到会话，后续仅进行SSE推送，跳过缓存与DB持久化");
        }

        RequestParams requestParams = chatSessionManager.getRequestParams(userId);
        String prompt = userChatMessage.getPrompt();
        requestParams.setConfirmed(userChatMessage.getConfirm());
        handlerData(requestParams, userChatMessage);

        return intentTool.getToolByIntent(prompt, requestParams, userChatMessage);
    }
}
