package pro.spss.server.agent.service.chatCore;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.mchange.v2.resourcepool.TimeoutException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientResponseException;
import pro.spss.server.agent.domain.constant.ChatConstants;
import pro.spss.server.agent.domain.entity.AIResponse;
import reactor.core.publisher.Mono;

import java.time.Duration;

/**
 * 底层与大模型交互的客户端，封装 WebClient 请求与错误处理（无状态），由上层传入配置
 */
@Slf4j
@Component
public class ChatModelClient {

    @Value("${agent.handler_result}")
    private boolean handlerResult;

    /**
     * 发送同步请求到大模型（由上层传入 url 与 key 配置）
     */
    public String sendRequestSync(JSONObject requestBody, ChatApiConfig config) {
        WebClient webClient = WebClient.builder()
                .baseUrl(config.getUrl())
                .defaultHeader("Authorization", "Bearer " + config.getKey())
                .defaultHeader("Content-Type", MediaType.APPLICATION_JSON_VALUE)
                .build();

        log.debug("发送给大模型的信息：");
        log.debug(requestBody.toString());
        long startTime = System.currentTimeMillis();
        try {
            String response = webClient.post()
                    .bodyValue(requestBody.toJSONString())
                    .retrieve()
                    .onStatus(this::isErrorStatus, this::handleErrorResponse)
                    .bodyToMono(String.class)
                    .timeout(Duration.ofSeconds(300))
                    .doOnSuccess(result -> log.info("大模型请求成功，耗时: {}ms",
                            System.currentTimeMillis() - startTime))
                    .doOnError(error -> log.error("大模型请求失败，耗时: {}ms",
                            System.currentTimeMillis() - startTime))
                    .block();
            log.debug("大模型返回");
            log.debug(response);
            return response != null ? response : "服务返回空响应";
        } catch (Exception e) {
            return formatExceptionMessage(e);
        }
    }

    private boolean isErrorStatus(HttpStatus status) {
        return !status.is2xxSuccessful();
    }

    private Mono<Throwable> handleErrorResponse(ClientResponse response) {
        HttpStatus statusCode = response.statusCode();
        return response.bodyToMono(String.class)
                .defaultIfEmpty("无错误详情")
                .map(errorBody -> {
                    String errorMessage = String.format("大模型服务异常 - 状态码: %d, 错误: %s",
                            statusCode.value(), truncateErrorMessage(errorBody));
                    if (statusCode.is4xxClientError()) {
                        return new IllegalArgumentException(errorMessage);
                    } else {
                        return new RuntimeException(errorMessage);
                    }
                })
                .flatMap(Mono::error);
    }

    private String truncateErrorMessage(String errorBody) {
        if (errorBody == null) return "无错误信息";
        return errorBody.length() > 200 ? errorBody.substring(0, 200) + "..." : errorBody;
    }

    private String formatExceptionMessage(Exception e) {
        log.error("大模型服务请求异常", e);
        if (e instanceof TimeoutException) {
            return "{\"error\": \"请求超时\", \"message\": \"服务响应超时，请稍后重试\"}";
        } else if (e instanceof WebClientResponseException.TooManyRequests) {
            return "{\"error\": \"请求过于频繁\", \"message\": \"请降低请求频率\"}";
        } else if (e instanceof WebClientResponseException) {
            WebClientResponseException wcre = (WebClientResponseException) e;
            return String.format("{\"error\": \"服务异常\", \"status\": %d, \"message\": \"%s\"}",
                    wcre.getStatusCode().value(), "远程服务响应异常");
        } else if (e.getCause() != null) {
            return String.format("{\"error\": \"服务异常\", \"message\": \"%s\"}",
                    e.getCause().getMessage());
        } else {
            return "{\"error\": \"系统异常\", \"message\": \"服务暂时不可用，请稍后重试\"}";
        }
    }

    /**
     * 从大模型返回的响应中解析 choices 数组
     */
    public JSONArray parseChoices(String responseBody) {
        if (responseBody == null) {
            return null;
        }
        JSONObject responseJson = JSONObject.parseObject(responseBody);
        return responseJson.getJSONArray("choices");
    }

    public AIResponse parseResponse(String responseBody) {
        if (responseBody == null) {
            return null;
        }
        JSONObject responseJson = JSONObject.parseObject(responseBody);
        JSONArray choices = responseJson.getJSONArray("choices");
        int totalToken = responseJson.getJSONObject("usage").getInteger("total_tokens");
        AIResponse aiResponse = new AIResponse();
        aiResponse.setTotalToken(totalToken);
        String responseText = "";
        String think = null;
        //结果处理
        if (choices != null && !choices.isEmpty()) {
            JSONObject assistantMessage = choices.getJSONObject(0).getJSONObject(ChatConstants.MESSAGE);
            String content = assistantMessage.getString(ChatConstants.CONTENT).replaceAll(ChatConstants.MARKDOWN, "");
            if(handlerResult) {
                responseText = content.substring(content.indexOf("</think>") + 10);
                int startIndex = content.indexOf("<think>") + 8;
                int endIndex = content.indexOf("</think>");
                if (startIndex >= 0 && endIndex > startIndex) {
                    think = content.substring(startIndex, endIndex);
                    //think字段需要去除多余的换行，连续多个换行只保留一个
                    think = think.replaceAll("\\n+", "\n").trim();
                }
            }

            aiResponse.setResponseText(responseText);
            aiResponse.setThink(think);
            log.debug("大模型回复：");
            log.debug(aiResponse.toString());
        }
        return aiResponse;
    }

    /**
     * 统一的重试发送封装：在最大尝试次数与总超时预算内进行重试，简单退避。
     * 返回 null 表示发送失败。
     */
    public String sendWithRetry(JSONObject requestBody,
                                ChatApiConfig config,
                                int maxAttempts,
                                long backoffMs,
                                long totalTimeoutMs) {
        long start = System.currentTimeMillis();
        int attempt = 0;
        while (attempt < Math.max(1, maxAttempts)) {
            attempt++;
            try {
                String resp = sendRequestSync(requestBody, config);
                if (resp != null && !resp.isBlank()) {
                    return resp;
                }
                log.warn("模型响应为空，尝试 {} / {}", attempt, maxAttempts);
            } catch (Exception e) {
                log.warn("调用模型失败，尝试 {} / {}，错误：{}", attempt, maxAttempts, e.getMessage());
            }
            // 检查总超时预算
            if (System.currentTimeMillis() - start >= totalTimeoutMs) {
                log.warn("发送总耗时超过预算 {} ms，终止重试", totalTimeoutMs);
                break;
            }
            try {
                Thread.sleep(Math.max(0, backoffMs));
            } catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                break;
            }
        }
        return null;
    }
}
