package pro.spss.server.agent.service.chatCore;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import pro.spss.server.agent.domain.constant.ChatConstants;
import pro.spss.server.agent.domain.entity.AIResponse;

import javax.annotation.Resource;
import java.util.Map;

/**
 * 大模型聊天助手服务：负责消息裁切、请求体构建、会话更新；底层交互由 ChatModelClient 完成
 */
@Slf4j
@Service
public class ChatHelper {

    @Value("${agent.api.model_name}")
    private String modelName;

    @Value("${agent.api.url}")
    private String apiUrl;

    @Value("${agent.api.key}")
    private String apiKey;

    @Resource
    private ChatModelClient chatModelClient;


    /**
     * 构建请求体
     */
    public JSONObject buildRequestBody(JSONArray messages) {
        JSONObject requestBody = new JSONObject();
        requestBody.put("model", modelName);
        requestBody.put("messages", messages);
        requestBody.put("temperature", ChatConstants.TEMPERATURE);
        return requestBody;
    }

    /**
     * 同步发送请求（阻塞） - 由 ChatModelClient 执行
     */
    public String sendRequestSync(JSONObject requestBody) {
        ChatApiConfig cfg = new ChatApiConfig(apiUrl, apiKey, modelName);
        return chatModelClient.sendRequestSync(requestBody, cfg);
    }

    /**
     * 解析响应结果，提取 choices 数组
     */
    public JSONArray parseResponse(String responseBody) {
        return chatModelClient.parseChoices(responseBody);
    }

    public AIResponse parseResponseText(String responseBody) {
        return chatModelClient.parseResponse(responseBody);
    }

    /**
     * 解析响应并更新会话，返回 contentJson
     */
    public JSONObject parseAndUpdateSession(String responseBody, JSONArray messages, String sessionId, Map<String, JSONArray> chatSessions) {
        try {
            JSONArray choices = parseResponse(responseBody);
            if (choices != null && !choices.isEmpty()) {
                JSONObject assistantMessage = choices.getJSONObject(0).getJSONObject(ChatConstants.MESSAGE);
                assistantMessage.remove("reasoning_content");
                messages.add(assistantMessage);
                chatSessions.put(sessionId, messages);

                String rawContent = assistantMessage.getString(ChatConstants.CONTENT);
                if (rawContent == null) {
                    log.warn("响应中不包含 content 字段");
                    return null;
                }
                String content1 = rawContent.replaceAll(ChatConstants.MARKDOWN, "");
                int thinkEnd = content1.indexOf("</think>");
                String content = thinkEnd >= 0 ? content1.substring(thinkEnd + 8) : content1; // 安全处理
                log.info("解析后的内容: {}", content);
                try {
                    return JSONObject.parseObject(content);
                } catch (Exception parseEx) {
                    log.error("解析 content 为 JSON 失败，返回原始字符串包装", parseEx);
                    JSONObject fallback = new JSONObject();
                    fallback.put("response", "非常抱歉，我还在不断的学习中，未能给您推荐出合适的算法，请您尝试将您的需求给我描述的再详细一点。");
                    fallback.put("options", new JSONArray());
                    fallback.put("params", new  JSONArray());
                    return fallback;
                }
            }
        } catch (Exception e) {
            log.error("解析响应并更新会话失败", e);
        }
        return null;
    }

    /**
     * 纯解析：从原始响应体中提取 content（去除 reasoning_content 与 markdown，截断 </think>），
     * 并尝试解析为 JSON；失败则返回带默认字段的 JSONObject（response/options/params）。
     * 无副作用，不更新会话缓存。
     */
    public JSONObject parseContentJson(String responseBody) {
        try {
            JSONArray choices = parseResponse(responseBody);
            if (choices != null && !choices.isEmpty()) {
                JSONObject assistantMessage = choices.getJSONObject(0).getJSONObject(ChatConstants.MESSAGE);
                // 去除可能存在的推理字段
                assistantMessage.remove("reasoning_content");
                String rawContent = assistantMessage.getString(ChatConstants.CONTENT);
                if (rawContent == null) {
                    log.warn("响应中不包含 content 字段");
                    return null;
                }
                // 去除 markdown 并截断 think
                String content1 = rawContent.replaceAll(ChatConstants.MARKDOWN, "");
                int thinkEnd = content1.indexOf("</think>");
                String content = thinkEnd >= 0 ? content1.substring(thinkEnd + 8) : content1;
                try {
                    return JSONObject.parseObject(content);
                } catch (Exception parseEx) {
                    log.error("解析 content 为 JSON 失败，返回默认结构", parseEx);
                    JSONObject fallback = new JSONObject();
                    fallback.put(ChatConstants.RESPONSE, "非常抱歉，我还在不断的学习中，未能给您推荐出合适的算法，请您尝试将您的需求给我描述的再详细一点。");
                    fallback.put(ChatConstants.OPTIONS, new JSONArray());
                    fallback.put(ChatConstants.PARAMS, new JSONArray());
                    return fallback;
                }
            }
        } catch (Exception e) {
            log.error("纯解析 contentJson 失败", e);
        }
        return null;
    }

    /**
     * 统一重试发送的入口：构造 ChatApiConfig 并委托给 ChatModelClient。
     */
    public String sendWithRetry(JSONObject requestBody, int maxAttempts, long backoffMs, long totalTimeoutMs) {
        ChatApiConfig cfg = new ChatApiConfig(apiUrl, apiKey, modelName);
        return chatModelClient.sendWithRetry(requestBody, cfg, maxAttempts, backoffMs, totalTimeoutMs);
    }
}
