package pro.spss.server.agent.utils;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;

/**
 * 根据算法ID与数据概要自动生成一份基础参数配置（默认值）。
 * 注意：这里的参数示例需要后续根据真实算法接口进行补充和调整。
 */
public class AlgoParamAutoGenerator {

    public static JSONArray generateDefaults(String algoId, String dataSummary) {
        JSONArray arr = new JSONArray();
        if (algoId == null) {
            return arr; // 无算法ID不生成
        }
        switch (algoId) {
            case "1": // 最小二乘回归 示例参数
                arr.add(makeParam("target", guessTargetColumn(dataSummary), "string", "因变量列名"));
                arr.add(makeParam("features", guessFeatureColumns(dataSummary), "array", "自变量列集合"));
                break;
            case "9": // 决策树 示例参数
                arr.add(makeParam("target", guessTargetColumn(dataSummary), "string", "分类目标列"));
                arr.add(makeParam("maxDepth", "6", "int", "最大树深度"));
                arr.add(makeParam("minSamplesSplit", "2", "int", "最小分裂样本数"));
                break;
            case "10": // 随机森林 示例参数
                arr.add(makeParam("target", guessTargetColumn(dataSummary), "string", "分类/回归目标列"));
                arr.add(makeParam("nEstimators", "100", "int", "树的数量"));
                arr.add(makeParam("maxFeatures", "auto", "string", "最大特征数策略"));
                break;
            case "13": // K均值聚类 示例参数
                arr.add(makeParam("k", "5", "int", "聚类簇数量"));
                arr.add(makeParam("init", "k-means++", "string", "初始化方法"));
                break;
            case "16": // 因子分析 示例参数
                arr.add(makeParam("nFactors", "3", "int", "提取因子数"));
                arr.add(makeParam("rotation", "varimax", "string", "旋转方法"));
                break;
            case "17": // 主成分分析 示例参数
                arr.add(makeParam("nComponents", "5", "int", "保留主成分数"));
                arr.add(makeParam("whiten", "false", "bool", "是否白化"));
                break;
            default:
                // 通用占位参数
                arr.add(makeParam("target", guessTargetColumn(dataSummary), "string", "目标列(如适用)"));
                arr.add(makeParam("features", guessFeatureColumns(dataSummary), "array", "特征列集合(如适用)"));
        }
        return arr;
    }

    private static JSONObject makeParam(String name, Object value, String type, String desc) {
        JSONObject obj = new JSONObject();
        obj.put("name", name);
        obj.put("value", value);
        obj.put("type", type);
        obj.put("desc", desc);
        return obj;
    }

    // 简单启发式：取概要首列为目标列（若包含"类型"或"分类"等文字可再优化）
    private static String guessTargetColumn(String dataSummary) {
        if (dataSummary == null) return "";
        String[] lines = dataSummary.split("\n");
        for (String l : lines) {
            if (l.matches("\\d+\\. .+\\(类型:.*")) {
                // 行格式: 序号. 列名 (类型:xxx) 示例:...
                int dotIdx = l.indexOf('.') + 1;
                int typeIdx = l.indexOf("(类型:");
                if (dotIdx > 0 && typeIdx > dotIdx) {
                    return l.substring(dotIdx, typeIdx).trim();
                }
            }
        }
        return "";
    }

    // 简单启发式：返回除目标列外的其余列名列表（逗号分隔字符串）
    private static String guessFeatureColumns(String dataSummary) {
        if (dataSummary == null) return "";
        String target = guessTargetColumn(dataSummary);
        StringBuilder sb = new StringBuilder();
        String[] lines = dataSummary.split("\n");
        for (String l : lines) {
            if (l.matches("\\d+\\. .+\\(类型:.*")) {
                int dotIdx = l.indexOf('.') + 1;
                int typeIdx = l.indexOf("(类型:");
                if (dotIdx > 0 && typeIdx > dotIdx) {
                    String col = l.substring(dotIdx, typeIdx).trim();
                    if (!col.equals(target)) {
                        if (sb.length() > 0) sb.append(',');
                        sb.append(col);
                    }
                }
            }
        }
        return sb.toString();
    }
}

