package pro.spss.server.agent.domain.response;

import lombok.extern.slf4j.Slf4j;
import org.springframework.core.MethodParameter;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyAdvice;
import pro.spss.server.agent.exception.BaseException;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;


/**
 * 结果封装：统一异常处理 + 结果包装。
 */
@Slf4j
@RestControllerAdvice
public class ResultWrapper implements ResponseBodyAdvice<Object> {

    private boolean isSseRequest(HttpServletRequest request, HttpServletResponse response) {
        try {
            String accept = request != null ? request.getHeader("Accept") : null;
            String contentType = response != null ? response.getContentType() : null;
            return (accept != null && accept.contains(MediaType.TEXT_EVENT_STREAM_VALUE))
                    || (contentType != null && contentType.contains(MediaType.TEXT_EVENT_STREAM_VALUE));
        } catch (Exception ignore) {
            return false;
        }
    }

    private ResponseEntity<String> buildSseErrorBody(int code, String message) {
        // SSE 文本事件格式：event + data + 空行
        String payload = new StringBuilder()
                .append("event: error\n")
                .append("data: {")
                .append("\"code\":").append(code).append(",")
                .append("\"message\":\"").append(message == null ? "" : message.replace("\n", " ")).append("\"}")
                .append("\n\n").toString();
        return ResponseEntity.ok()
                .contentType(MediaType.TEXT_EVENT_STREAM)
                .body(payload);
    }

    /** 全局异常处理 */
    @ExceptionHandler(Exception.class)
    Object handleAllException(HttpServletRequest request, HttpServletResponse response, Exception e) {
        log.error("出现了异常！", e);
        String message = (e == null) ? "未知的错误" : e.getMessage();
        // SSE 请求：按 SSE 事件返回，避免 JSON 转换器与 event-stream 冲突
        if (isSseRequest(request, response)) {
            return buildSseErrorBody(500, "服务异常：" + message);
        }
        // 常规请求：统一 Result JSON
        response.setStatus(200);
        return Result.failed(500, "服务异常：" + message);
    }

    /** 自定义业务异常处理 */
    @ExceptionHandler(BaseException.class)
    Object handleBaseException(HttpServletRequest request, HttpServletResponse response, Exception e) {
        log.error("出现了异常！", e);
        BaseException be = (BaseException) e;
        if (isSseRequest(request, response)) {
            return buildSseErrorBody(be.getCode(), be.getErrorMessage());
        }
        response.setStatus(200);
        return Result.failed(be.getCode(), be.getErrorMessage());
    }

    @Override
    public boolean supports(MethodParameter returnType, Class<? extends HttpMessageConverter<?>> converterType) {
        return true;
    }

    /** 结果统一处理：SSE 不做包装，直接透传。 */
    @Override
    public Object beforeBodyWrite(Object body, MethodParameter returnType, MediaType selectedContentType,
                                  Class<? extends HttpMessageConverter<?>> selectedConverterType, ServerHttpRequest request,
                                  ServerHttpResponse response) {
        if (selectedContentType != null && MediaType.TEXT_EVENT_STREAM.includes(selectedContentType)) {
            return body;
        }
        if (body instanceof Result) {
            return body;
        }
        return Result.success(body);
    }
}
